/*----------------------------------------------------------------------
MODULE lf.sv
A charge-pump loop filter in XMODEL
----------------------------------------------------------------------*/

`include "xmodel.h"

module lf #(
    parameter real init_value = 0.5,    // initial value
    parameter real I_up = 21e-6,        // up current
    parameter real I_dn = 20e-6,        // down current
    parameter real Rs = 20e3,           // series resistance of loop filter
    parameter real Cs = 500e-15,        // series capacitance of loop filter
    parameter real Cp = 50e-15          // shunt capacitance of loop filter
)(
    output xreal out,                   // output voltage
    input xbit up,                      // up input from pfd
    input xbit dn                       // dn input from pfd
);

    // filter parameters
    parameter real gain = 1.0/(Cs+Cp);
    parameter real zero = 1.0/Rs/Cs;
    parameter real pole = (1.0/Rs/Cs+1.0/Rs/Cp);

    xreal iup, idn;
    xreal v0, vup, vdn;

    transition  #(.value0(0.0), .value1(I_up))
                cp_up(.out(iup), .in(up));
    transition  #(.value0(0.0), .value1(-1.0*I_dn))
                cp_dn(.out(idn), .in(dn));
    filter      #(.gain(gain), .poles('{pole/(2*M_PI),0.0,0.0,0.0}), .zeros('{zero/(2*M_PI),0.0}))
                cp_filter1(.out(vup), .in(iup));
    filter      #(.gain(gain), .poles('{pole/(2*M_PI),0.0,0.0,0.0}), .zeros('{zero/(2*M_PI),0.0}))
                cp_filter2(.out(vdn), .in(idn));
    dc_gen      #(.value(init_value)) init_gen(.out(v0));
    add         #(.num_in(3), .scale('{1.0,1.0,1.0})) add(.in({vup, vdn, v0}), .out(out));

endmodule
