/*----------------------------------------------------------------------
MODULE lf.sv
A charge-pump loop filter in Real-Number Verilog
----------------------------------------------------------------------*/

module lf #(
    parameter real init_value = 0.5,    // initial value
    parameter real I_up = 21e-6,        // up current
    parameter real I_dn = 20e-6,        // down current
    parameter real Rs = 20e3,           // series resistance of loop filter
    parameter real Cs = 500e-15,        // series capacitance of loop filter
    parameter real Cp = 50e-15          // shunt capacitance of loop filter
)(
    output real out,                    // output voltage
    input up,                           // up input from pfd
    input dn                            // dn input from pfd
);

    // filter parameters
    parameter real gain = 1.0/(Cs+Cp);
    parameter real zero = 1.0/Rs/Cs;
    parameter real pole = (1.0/Rs/Cs+1.0/Rs/Cp);

    real inet = 0.0;
    real iacc = 0.0;
    real vout = 0.0;
    real dt = `ABS_UNIT;

    always begin
        #1;
        inet = up*I_up - dn*I_dn;
        vout = (gain*dt*(iacc + inet + inet/(zero*dt))*(pole*dt) + vout)/(1.0 + pole*dt);
        iacc += inet;
    end

    assign out = vout + init_value;

endmodule
